<?php

namespace Modules\Markup\Http\Controllers;

use Carbon\Carbon;
use GuzzleHttp\Client;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Modules\Markup\Entities\Markup;
use Modules\Users\Entities\User;

class MarkupController extends Controller
{
    public function store(Request $req, $type)
    {
        if (!in_array($type, $this->getMarkupTypes())) {
            return response()->json([
                'status' => false,
                'message' => 'Invalid Request'
            ], 400);
        }

        $validatedData = $req->validate(
            $this->getValidationRules()
        );
        $validatedData = $this->prepareData($validatedData);
        // return $validatedData;
        $validatedData['model_type'] = $type;
        if ($validatedData['type'] == 'amount') {
            $validatedData['percentage_b2c'] = 0;
            $validatedData['percentage_b2b'] = 0;
            $validatedData['percentage_mobile'] = 0;
        } else if ($validatedData['type'] == 'percentage') {
            $validatedData['amount_b2c'] = 0;
            $validatedData['amount_b2b'] = 0;
            $validatedData['amount_mobile'] = 0;
        }
        if (is_b2b_admin($req)) {
            $companyId = $req->company_id ?? $req->get('parent');
            $companyType = $req->markup_type == 'b2b_on_company' ? 'b2b_on_company' : 'b2b';
        } else {
            $companyId = $req->get('parent');
            if (!$companyId) {
                return json_response([], 401, false, "Forbidden");
            } else {
                $companyType = 'company';
            }
        }
        $validatedData['company_id'] = $companyId;
        $validatedData['company_type'] = $companyType;


        $markup = Markup::where([
            'model_id' => $validatedData['model_id'],
            'model_type' => $type,
            'company_id' => $validatedData['company_id'],
            'company_type' => $validatedData['company_type'],
        ])->first();
        // return $markup;
        if (!$markup) {
            $markup = Markup::create($validatedData);
        } else {
            $markup->fill($validatedData);
            $markup->save();
        }

        return response()->json([
            'status' => true,
            'message' => 'success'
        ]);
    }

    public function update($id, Request $req)
    {
        $validatedData = $req->validate(
            $this->getValidationRules()
        );

        $validatedData = $this->prepareData($validatedData);

        $markup = Markup::findOrFail($id);
        $markup->fill($validatedData);
        $markup->save();

        return response()->json([
            'status' => true,
            'message' => 'success'
        ]);
    }

    // public function delete($id)
    // {
    //     $markup = Markup::findOrFail($id);
    //     $markup->delete();

    //     return response()->json([
    //         'status' => true,
    //         'message' => 'success'
    //     ]);
    // }

    public function delete($type, $id, Request $req)
    {
        if (!in_array($type, $this->getMarkupTypes())) {
            return response()->json([
                'status' => false,
                'message' => 'Invalid Request'
            ], 400);
        }
        $markup = Markup::where([
            'model_id' => $id,
            'model_type' => $type,
            'company_id' => $req->get('parent')
        ]);
        $markup->delete();

        return response()->json([
            'status' => true,
            'message' => 'success'
        ]);
    }

    public function show($type, $id, Request $req)
    {
        $markup_type = $req->markup_type;
        $company_id = $req->company_id ??  $req->get('parent');
        $q = new Markup();
        if ($markup_type == 'b2b_on_company') {
            $q = $q->where('company_type', 'b2b_on_company');
        } else {
            $q = $q->where('company_type', '!=', 'b2b_on_company');
        }
        $markup = $q->where([
            'model_type' => $type,
            'model_id' => $id,
            'company_id' => $company_id,
        ])->first();
        if ($markup) {
            return response()->json([
                'status' => true,
                'data' => [
                    'markup' => $markup
                ]
            ], 200);
        } else {
            return json_response([]);
        }
    }

    protected function getValidationRules($extra = [])
    {
        $rules = [
            'model_name' => 'required',
            'model_id' => 'required',
            'type' => 'required',
            'amount_b2c' => 'required_if:type,amount',
            'amount_b2b' => 'required_if:type,amount',
            'amount_mobile' => 'required_if:type,amount',
            'percentage_b2c' => 'required_if:type,percentage',
            'percentage_b2b' => 'required_if:type,percentage',
            'percentage_mobile' => 'required_if:type,percentage',
            'max' => 'nullable',
            'min' => 'nullable',
            'ongoing' => 'nullable',
            'reservation_from' => 'required_if:ongoing, false',
            'reservation_to' => 'required_if:ongoing, false',
            'booking_from' => 'required_if:ongoing, false',
            'booking_to' => 'required_if:ongoing, false',
            'overwrite' => 'nullable|boolean',
            'identifier' => 'required_if:type,hotel'
        ];

        if (!empty($extra)) {
            return array_merge($rules, $extra);
        } else {
            return $rules;
        }
    }

    protected function getMarkupTypes()
    {
        return [
            'hotel',
            'module',
            'supplier',
            'country',
            'city',
            'airline',
            'airport',
            'airline_destination',
            'company',
            'employee'
        ];
    }

    protected function prepareData($data)
    {
        if (!$data['ongoing']) {
            $data['ongoing'] = 0;
            $data['reservation_from'] = Carbon::parse($data['reservation_from']);
            $data['reservation_to'] = Carbon::parse($data['reservation_to']);
            $data['booking_from'] = Carbon::parse($data['booking_from']);
            $data['booking_to'] = Carbon::parse($data['booking_to']);
        } else {
            $data['ongoing'] = 1;
            $data['reservation_from'] = null;
            $data['reservation_to'] = null;
            $data['booking_from'] = null;
            $data['booking_to'] = null;
        }

        return $data;
    }

    protected function getClient($isBase = false)
    {
        $headers = [
            'X-Authorization-Token' =>  request()->bearerToken(),
            'X-Access-Token' =>  env('API_TOKEN'),
            'X-Language-Code' => request()->header('X-Language-Code') ?? 'ar',
            'X-Country-Code' => request()->header('X-Country-Code') ?? 'SA',
            'X-Currency-Code' => request()->header('X-Currency-Code') ?? 'SAR',
            'X-Client-Ip' => request()->ip(),
            'Content-Type' => 'application/json',
            'Accept' => 'application/json'
        ];

        if (request()->header('X-Session')) {
            $headers['X-Session'] = request()->header('X-Session');
        }
        $client = new Client([
            'base_uri' => $isBase ? env('API_BASE') : env('API_MODULE_BASE'),
            'headers' => $headers
        ]);

        return $client;
    }

    // public function request($method, $urlPath, $data = []) {
    //     $client = $this->getClient();
    //     try {
    //         $response = $client->request($method, $urlPath, $data);
    //     } catch (\Throwable $th) {
    //         return response()->json([
    //             'status' => false,
    //             'message' => 'Internal Server Error'
    //         ], 500);
    //     }
    //     return $response;
    // }


    public function getModules()
    {
        // return env('API_MODULE_BASE');
        $client = $this->getClient();
        try {
            $response = $client->get('users/user_permissions');
        } catch (\Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Internal Server Error'
            ], 500);
        }
        $response = json_decode($response->getBody()->getContents());
        // return $response->data->Permissions;
        if ($response->status) {
            $modules = [];
            foreach ($response->data->Permissions as $module) {
                $moduleData = [
                    'moduleId' => (int) $module->ModuleId,
                    'moduleName' => $module->ModuleName,
                    'suppliers' => []
                ];
                if (!empty($module->Suppliers)) {
                    $suppliers = [];
                    foreach ($module->Suppliers as $supplier) {
                        $suppliers[] = [
                            'supplierId' => (int) $supplier->SupplierId,
                            'supplierName' => $supplier->SupplierName
                        ];
                    }
                    $moduleData['suppliers'] = $suppliers;
                }
                $modules[] = $moduleData;
            }
        }
        return response()->json([
            'status' => true,
            'data' => [
                'modules' => $modules
            ]
        ]);
    }

    public function getCountries($query)
    {
        $client = $this->getClient(true);
        try {
            $response = $client->get("base/country_list?q={$query}");
        } catch (\Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Internal Server Error'
            ], 500);
        }
        $response = json_decode($response->getBody()->getContents());
        return response()->json([
            'status' => true,
            'data' => [
                'countries' => $response->data
            ]
        ]);
    }

    public function getHotels($cityCode, $query)
    {
        if (!$query or strlen($query) < 3) {
            return response()->json([
                'status' => false,
                'message' => 'minimum search length is 3 characters'
            ], 200);
        }
        $client = $this->getClient(true);
        // return "hotels/hotel_list/?city_code={$cityCode}&q={$query}";
        try {
            $response = $client->get("base/hotel_list/?city_code={$cityCode}&q={$query}", request()->all());
        } catch (\Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Internal Server Error'
            ], 500);
        }
        // return "jjjjj";
        // return $response;
        $hotels = [];
        $response = json_decode($response->getBody()->getContents());
        foreach ($response->data as $hotel) {
            if (!empty($hotel->HotelCode)) {
                $hotels[] = $hotel;
            }
        }
        return response()->json([
            'status' => true,
            'data' => [
                'hotels' => $hotels
            ]
        ]);
    }

    public function getCities($countryCode, $query)
    {
        if (strlen($query) < 3) {
            return response()->json([
                'status' => false,
                'message' => 'minimum search length is 3 characters'
            ], 400);
        }
        $client = $this->getClient(true);
        try {
            $response = $client->get("base/city_list/?country_code={$countryCode}&q={$query}");
        } catch (\Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Internal Server Error'
            ], 500);
        }
        $response = json_decode($response->getBody()->getContents());
        return response()->json([
            'status' => true,
            'data' => [
                'cities' => $response->data
            ]
        ]);
    }


    public function getAirlines($query)
    {
        $client = $this->getClient(true);
        try {
        } catch (\Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Internal Server Error'
            ], 500);
        }
        $response = $client->get("base/airline_list/?q={$query}");
        // return $response;
        $response = json_decode($response->getBody()->getContents());
        return response()->json([
            'status' => true,
            'data' => [
                'airlines' => $response->data
            ]
        ]);
    }

    public function getAirports($query)
    {
        $client = $this->getClient(true);
        try {
        } catch (\Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Internal Server Error'
            ], 500);
        }
        $response = $client->get("base/airport_list/?q={$query}");
        // return $response;
        $response = json_decode($response->getBody()->getContents());
        return response()->json([
            'status' => true,
            'data' => [
                'airports' => $response->data
            ]
        ]);
    }

    public function getAirlineDestination()
    {
        $client = $this->getClient();
        try {
        } catch (\Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => 'Internal Server Error'
            ], 500);
        }
        $response = $client->get('hotels/country_list');
        // return $response;
        $response = json_decode($response->getBody()->getContents());
        return response()->json([
            'status' => true,
            'data' => [
                'destinations' => $response->data
            ]
        ]);
    }

    public function list($type, Request $req)
    {
        $markups = Markup::where([
            'model_type' => $type,
            'company_id' => $req->get('parent')
        ])->get();

        return json_response([
            'markups' => $markups->all()
        ], 200, true, 'success');
    }

    public function getCompanies(Request $req)
    {
        if (!is_b2b_admin($req)) {
            return json_response([], 200, false, 'Unauthorized');
        }

        $companies = User::select(['id', 'company'])->where(['type' => 'company'])->get();
        return json_response([
            'companies' => $companies->all()
        ]);
    }

    public function getEmployees(Request $req, $id = null)
    {
        if (!$id) {
            $parent = $req->get('parent');
        } else {
            $parent = $id;
        }

        $employees = User::where(['parent_id' => $parent, 'type' => 'employee'])->get();

        $operationsEmployees = [];

        foreach ($employees as $employee) {
            if ($employee->hasRole('operation')) {
                $emp = $employee->only(['id', 'first_name', 'last_name', 'email']);
                $emp['name'] = $employee->first_name . ' ' . $employee->last_name;
                $operationsEmployees[] = $emp;
            }
        }

        return json_response([
            'employees' => $operationsEmployees
        ]);
    }

    public function getCompanyModules($id)
    {
        $user = User::findOrFail($id);
        $user_permissions = $user->permissions;

        $suppliers = [];
        $temp_modules = [];
        $modules = [];
        if ($user_permissions) {
            foreach ($user_permissions as $row) {
                $module_name = ucfirst(explode('_', $row['name'])[0]);
                if (!in_array($module_name, $temp_modules, true)) {
                    array_push($temp_modules, $module_name);
                }
                if (strpos($row['name'], '.') !== false) {
                    $module_name = ucfirst(explode('_', $row['name'])[0]);
                    $supplier = ucfirst(explode('_', $row['name'])[1]);
                    $suppliers[$module_name][] =
                        [
                            'supplierName' => str_replace('_', " ", explode('.', $supplier)[0]),
                            'supplierId' => explode('.', $supplier)[1],
                        ];
                }
            }

            //get modules ids from backoffice
            $backofficeModules = $this->getBackOfficeModules();
            foreach ($backofficeModules as $key => $val) {
                if (in_array($key, $temp_modules, true)) {
                    $modules[] = [
                        'moduleName' => $key,
                        'moduleId' => $val,
                        'suppliers' => isset($suppliers[$key]) ? $suppliers[$key] : []
                    ];
                }
            }
        }

        return response()->json([
            'status' => true,
            'data' => [
                'modules' => $modules
            ]
        ]);
    }
    private function getBackOfficeModules()
    {
        //$backofficeModules = json_decode($this->getModules()->getContent())->data->modules;
        $modules = ['Flights','Hotels','Payments'];
        //foreach ($backofficeModules as $module) {
          //  $modules[$module->moduleName] = $module->moduleId;
        //}
        return $modules;
    }
}
