"use strict";Object.defineProperty(exports, "__esModule", {value: true}); function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }// src/index.ts
var _path = require('path');
var _fs = require('fs'); var _fs2 = _interopRequireDefault(_fs);
var _workboxbuild = require('workbox-build');

// src/html.ts


// src/constants.ts
var FILE_MANIFEST = "manifest.webmanifest";
var FILE_SW_REGISTER = "registerSW.js";

// src/html.ts
function join(...args) {
  return _path.join.call(void 0, ...args).replace(/\\/g, "/");
}
function generateSWRegister(options) {
  return `
if('serviceWorker' in navigator) {
window.addEventListener('load', () => {
navigator.serviceWorker.register('${join(options.base, options.filename)}', { scope: '${options.scope}' })
})
}`.replace(/\n/g, "");
}
function injectServiceWorker(html, options) {
  if (options.injectRegister === "inline") {
    return html.replace("</head>", `
<link rel="manifest" href="${join(options.base, FILE_MANIFEST)}">
<script>${generateSWRegister(options)}</script>
</head>`.trim());
  } else if (options.injectRegister === "import") {
    return html.replace("</head>", `
<link rel="manifest" href="${join(options.base, FILE_MANIFEST)}">
<script src="${join(options.base, FILE_SW_REGISTER)}"></script>
</head>`.trim());
  } else {
    return html.replace("</head>", `
<link rel="manifest" href="${join(options.base, FILE_MANIFEST)}">
</head>`.trim());
  }
}

// src/config.ts



// src/cache.ts
var cachePreset = [
  {
    urlPattern: /^https:\/\/fonts\.(?:googleapis|gstatic)\.com\/.*/i,
    handler: "CacheFirst",
    options: {
      cacheName: "google-fonts",
      expiration: {
        maxEntries: 4,
        maxAgeSeconds: 365 * 24 * 60 * 60
      }
    }
  },
  {
    urlPattern: /\.(?:eot|otf|ttc|ttf|woff|woff2|font.css)$/i,
    handler: "StaleWhileRevalidate",
    options: {
      cacheName: "static-font-assets",
      expiration: {
        maxEntries: 4,
        maxAgeSeconds: 7 * 24 * 60 * 60
      }
    }
  },
  {
    urlPattern: /\.(?:jpg|jpeg|gif|png|svg|ico|webp)$/i,
    handler: "StaleWhileRevalidate",
    options: {
      cacheName: "static-image-assets",
      expiration: {
        maxEntries: 64,
        maxAgeSeconds: 24 * 60 * 60
      }
    }
  },
  {
    urlPattern: /\.(?:js)$/i,
    handler: "StaleWhileRevalidate",
    options: {
      cacheName: "static-js-assets",
      expiration: {
        maxEntries: 32,
        maxAgeSeconds: 24 * 60 * 60
      }
    }
  },
  {
    urlPattern: /\.(?:css|less)$/i,
    handler: "StaleWhileRevalidate",
    options: {
      cacheName: "static-style-assets",
      expiration: {
        maxEntries: 32,
        maxAgeSeconds: 24 * 60 * 60
      }
    }
  },
  {
    urlPattern: /\.(?:json|xml|csv)$/i,
    handler: "NetworkFirst",
    options: {
      cacheName: "static-data-assets",
      expiration: {
        maxEntries: 32,
        maxAgeSeconds: 24 * 60 * 60
      }
    }
  },
  {
    urlPattern: /\/api\/.*$/i,
    handler: "NetworkFirst",
    method: "GET",
    options: {
      cacheName: "apis",
      expiration: {
        maxEntries: 16,
        maxAgeSeconds: 24 * 60 * 60
      },
      networkTimeoutSeconds: 10
    }
  },
  {
    urlPattern: /.*/i,
    handler: "NetworkFirst",
    options: {
      cacheName: "others",
      expiration: {
        maxEntries: 32,
        maxAgeSeconds: 24 * 60 * 60
      },
      networkTimeoutSeconds: 10
    }
  }
];

// src/config.ts
function resolveBathPath(base) {
  if (isAbsolute(base))
    return base;
  return !base.startsWith("/") && !base.startsWith("./") ? `/${base}` : base;
}
function isAbsolute(url) {
  return url.match(/^(?:[a-z]+:)?\/\//i);
}
function resolveOptions(options, viteConfig) {
  const root = viteConfig.root;
  const pkg = _fs2.default.existsSync("package.json") ? JSON.parse(_fs2.default.readFileSync("package.json", "utf-8")) : {};
  const {
    mode = process["env"]["NODE_ENV"] || "production",
    srcDir = "public",
    outDir = viteConfig.build.outDir || "dist",
    injectRegister = "import",
    filename = "sw.js",
    strategies = "generateSW",
    minify = true,
    base = viteConfig.base
  } = options;
  const basePath = resolveBathPath(base);
  const swSrc = _path.resolve.call(void 0, root, srcDir, filename);
  const swDest = _path.resolve.call(void 0, root, outDir, filename);
  const outDirRoot = _path.resolve.call(void 0, root, outDir);
  const scope = options.scope || basePath;
  const defaultWorkbox = {
    swDest,
    globDirectory: outDirRoot,
    offlineGoogleAnalytics: false,
    runtimeCaching: cachePreset,
    mode,
    navigateFallback: "index.html"
  };
  const defaultInjectManifest = {
    swSrc,
    swDest,
    globDirectory: outDirRoot,
    injectionPoint: "self.__WB_MANIFEST"
  };
  const defaultManifest = {
    name: pkg.name,
    short_name: pkg.name,
    start_url: basePath,
    display: "standalone",
    background_color: "#ffffff",
    lang: "en",
    scope
  };
  const workbox = Object.assign({}, defaultWorkbox, options.workbox || {});
  const manifest = Object.assign({}, defaultManifest, options.manifest || {});
  const injectManifest2 = Object.assign({}, defaultInjectManifest, options.injectManifest || {});
  return {
    base: basePath,
    mode,
    swDest,
    srcDir,
    outDir,
    injectRegister,
    filename,
    strategies,
    workbox,
    manifest,
    injectManifest: injectManifest2,
    scope,
    minify
  };
}

// src/index.ts
function VitePWA(userOptions = {}) {
  let viteConfig;
  let options;
  return {
    name: "vite-plugin-pwa",
    enforce: "post",
    apply: "build",
    configResolved(config) {
      viteConfig = config;
      options = resolveOptions(userOptions, viteConfig);
    },
    transformIndexHtml: {
      enforce: "post",
      transform(html) {
        return injectServiceWorker(html, options);
      }
    },
    generateBundle(_, bundle) {
      bundle[FILE_MANIFEST] = {
        isAsset: true,
        type: "asset",
        name: void 0,
        source: `${JSON.stringify(options.manifest, null, options.minify ? 0 : 2)}
`,
        fileName: FILE_MANIFEST
      };
      if (options.injectRegister === "import" && !_fs.existsSync.call(void 0, _path.join.call(void 0, viteConfig.root, "public", FILE_SW_REGISTER))) {
        bundle[FILE_SW_REGISTER] = {
          isAsset: true,
          type: "asset",
          name: void 0,
          source: generateSWRegister(options),
          fileName: FILE_SW_REGISTER
        };
      }
    },
    async closeBundle() {
      if (!viteConfig.build.ssr) {
        if (options.strategies === "injectManifest")
          await _workboxbuild.injectManifest.call(void 0, options.injectManifest);
        else
          await _workboxbuild.generateSW.call(void 0, options.workbox);
      }
    }
  };
}


exports.VitePWA = VitePWA;
