<?php

namespace Modules\StaticData\Entities;

use Astrotomic\Translatable\Translatable;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Hotel extends Model
{
    use HasFactory, Translatable;

    protected $table = 'hotels';

    protected $fillable = [
        'city_id',
        'country_id',
        'lat',
        'lng',
        'phone',
        'fax',
        'website',
        'email',
        'main_image',
        'gallery',
        'facilities',
        'online_tbo_gallery',
        'pincode',
        'rating',
        'tripadvisor_rating',
        'tripadvisor_review_url',
        'weight',
        'is_deleted',
        'published',
    ];

    protected $casts = [
        'gallery' => 'array',
        'facilities' => 'array',
    ];

    public $translatedAttributes = [
        'name',
        'room_facilities',
        'attractions',
        'services',
        'policy',
        'description',
        'address',
    ];

    protected $appends = ['hotel_city'];

    public function city()
    {
        return $this->belongsTo(City::class, 'city_id');
    }

    public function getFacilitiesAttribute()
    {
        $currentLocale = request()->header('X-Language-Code', 'default_locale');
        $facilityIds = explode(',', $this->attributes['facilities']);

        $facilities = HotelFacility::whereIn('id', $facilityIds)->get();

        return $facilities->map(function ($facility) use ($currentLocale) {
            return [
                'icon' => $facility->icon,
                'name' => $facility->translate($currentLocale)->name,
            ];
        });
    }

    public function getHotelCityAttribute()
    {
        return $this->city ?? null; // Use the "city" relationship directly
    }

    public function getGalleryAttribute($value)
    {
        $localFiles = false;

        $galleryLinks = json_decode($value, true);
        if (!is_array($galleryLinks)) {
            return [];
        }
        $lastThreeExternalLinks = array_slice($galleryLinks, -7);
        return $lastThreeExternalLinks;

        $workingImageLinks = [];

        foreach ($galleryLinks as $link) {
            if (!str_starts_with($link, 'https://')) {
                // If it's not an external link (doesn't start with 'https://')
                // Check if the local file exists
                if (file_exists(public_path($link))) {
                    $workingImageLinks[] = $link;
                    $localFiles = true;
                }
            }

            if (count($workingImageLinks) >= 3) {
                break; // Stop after finding the first 3 working image links
            }
        }

        // If there are local files, return them
        if ($localFiles) {
            return $workingImageLinks;
        }

        // Otherwise, fetch external links (up to 3)
        foreach ($galleryLinks as $link) {
            if (str_starts_with($link, 'https://')) {
                $headers = get_headers($link);

                // Check if the link returns a "200 OK" status
                if (in_array('HTTP/1.1 200 OK', $headers)) {
                    $workingImageLinks[] = $link;
                }

                if (count($workingImageLinks) >= 3) {
                    break; // Stop after finding the first 3 working external image links
                }
            }
        }

        return $workingImageLinks;
    }

    public function supplierHotels()
    {
        return $this->hasMany(SupplierHotel::class, 'hotel_id', 'id');
    }

    public static function getHotelsBySupplierCode($supplierId, $hotelCode)
    {
        return static::whereHas('supplierHotels', function ($query) use ($supplierId, $hotelCode) {
            $query->where('supplier_id', $supplierId)
                ->where('code', $hotelCode);
        })->get();
    }
}
